#include <Arduino.h>
#include <arduino-timer.h>
#include <TM1637TinyDisplay6.h>
#include <Adafruit_NeoPixel.h>
#include <OneButton.h>


#define DEBUG false
#define PIN_AUTO_FIRE_LEVEL 34
#define PIN_DISPLAY_CLK 18
#define PIN_DISPLAY_DIO 19
#define PIN_LED_STRIP 32

#define PIN_BUTTON_LED_TOGGLE_AF 23
#define PIN_BUTTON_LED_TOGGLE_MENU 21
#define PIN_BUTTON_LED_FIRE 22

#define PIN_BUTTON_TOGGLE_AF 16
#define PIN_BUTTON_TOGGLE_MENU 17

#define PIN_RPI_TOGGLE_MENU 33
#define PIN_RPI_FIRE 13


#define LED_COUNT 10
#define AUTO_FIRE_MIN_INTERVAL 1000
#define AUTO_FIRE_MAX_INTERVAL 125
#define AUTO_FIRE_RESOLUTION 4095

#define LED_MAX_BRIGHTNESS 200

auto timer = timer_create_default();
TM1637TinyDisplay6 display(PIN_DISPLAY_CLK, PIN_DISPLAY_DIO);

Adafruit_NeoPixel led = Adafruit_NeoPixel(LED_COUNT, PIN_LED_STRIP, NEO_GRB + NEO_KHZ800);

OneButton btnToggleAf = OneButton(PIN_BUTTON_TOGGLE_AF, false, false);
OneButton btnToggleMenu = OneButton(PIN_BUTTON_TOGGLE_MENU, false, false);

/* Animation Data - HGFEDCBA Map */
const uint8_t ANIMATION[58][6] = {
  { 0x40, 0x00, 0x00, 0x00, 0x00, 0x00 },  // Frame 0
  { 0x49, 0x40, 0x00, 0x00, 0x00, 0x00 },  // Frame 1
  { 0x49, 0x49, 0x40, 0x00, 0x00, 0x00 },  // Frame 2
  { 0x49, 0x49, 0x49, 0x40, 0x00, 0x00 },  // Frame 3
  { 0x49, 0x49, 0x49, 0x49, 0x40, 0x00 },  // Frame 4
  { 0x49, 0x49, 0x49, 0x49, 0x49, 0x40 },  // Frame 5
  { 0x09, 0x49, 0x49, 0x49, 0x49, 0x49 },  // Frame 6
  { 0x00, 0x09, 0x49, 0x49, 0x49, 0x49 },  // Frame 7
  { 0x00, 0x00, 0x09, 0x49, 0x49, 0x49 },  // Frame 8
  { 0x00, 0x00, 0x00, 0x09, 0x49, 0x49 },  // Frame 9
  { 0x00, 0x00, 0x00, 0x00, 0x09, 0x49 },  // Frame 10
  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x09 },  // Frame 11
  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },  // Frame 12
  { 0x20, 0x00, 0x00, 0x00, 0x00, 0x02 },  // Frame 13
  { 0x32, 0x00, 0x00, 0x00, 0x00, 0x26 },  // Frame 14
  { 0x36, 0x20, 0x00, 0x00, 0x02, 0x36 },  // Frame 15
  { 0x36, 0x32, 0x00, 0x00, 0x26, 0x36 },  // Frame 16
  { 0x36, 0x36, 0x20, 0x02, 0x36, 0x36 },  // Frame 17
  { 0x36, 0x36, 0x32, 0x26, 0x36, 0x36 },  // Frame 18
  { 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 },  // Frame 19
  { 0x3f, 0x36, 0x36, 0x36, 0x36, 0x3f },  // Frame 20
  { 0x3f, 0x3f, 0x36, 0x36, 0x3f, 0x3f },  // Frame 21
  { 0x7f, 0x3f, 0x3f, 0x3f, 0x3f, 0x7f },  // Frame 22
  { 0x7f, 0x7f, 0x3f, 0x3f, 0x7f, 0x7f },  // Frame 23
  { 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f },  // Frame 24
  { 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f },  // Frame 25
  { 0x77, 0x7f, 0x7f, 0x7f, 0x7f, 0x7e },  // Frame 26
  { 0x76, 0x77, 0x7f, 0x7f, 0x7e, 0x76 },  // Frame 27
  { 0x36, 0x76, 0x77, 0x7e, 0x76, 0x36 },  // Frame 28
  { 0x36, 0x36, 0x76, 0x76, 0x36, 0x36 },  // Frame 29
  { 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 },  // Frame 30
  { 0x26, 0x36, 0x36, 0x36, 0x36, 0x34 },  // Frame 31
  { 0x06, 0x26, 0x36, 0x36, 0x36, 0x10 },  // Frame 32
  { 0x06, 0x26, 0x32, 0x26, 0x36, 0x10 },  // Frame 33
  { 0x06, 0x24, 0x12, 0x26, 0x22, 0x10 },  // Frame 34
  { 0x02, 0x24, 0x12, 0x24, 0x22, 0x10 },  // Frame 35
  { 0x02, 0x04, 0x02, 0x20, 0x02, 0x10 },  // Frame 36
  { 0x02, 0x00, 0x02, 0x20, 0x00, 0x10 },  // Frame 37
  { 0x02, 0x00, 0x00, 0x20, 0x00, 0x10 },  // Frame 38
  { 0x02, 0x00, 0x00, 0x20, 0x00, 0x00 },  // Frame 39
  { 0x00, 0x00, 0x00, 0x20, 0x00, 0x00 },  // Frame 40
  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },  // Frame 41
  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 },  // Frame 42
  { 0x80, 0x80, 0x00, 0x00, 0x00, 0x00 },  // Frame 43
  { 0x80, 0x80, 0x80, 0x00, 0x00, 0x00 },  // Frame 44
  { 0x80, 0x80, 0x80, 0x80, 0x00, 0x00 },  // Frame 45
  { 0x00, 0x80, 0x80, 0x80, 0x80, 0x00 },  // Frame 46
  { 0x00, 0x00, 0x80, 0x80, 0x80, 0x80 },  // Frame 47
  { 0x00, 0x00, 0x00, 0x80, 0x80, 0x80 },  // Frame 48
  { 0x00, 0x00, 0x00, 0x00, 0x80, 0x80 },  // Frame 49
  { 0x00, 0x00, 0x00, 0x80, 0x80, 0x80 },  // Frame 50
  { 0x00, 0x00, 0x80, 0x80, 0x80, 0x80 },  // Frame 51
  { 0x00, 0x80, 0x80, 0x80, 0x80, 0x00 },  // Frame 52
  { 0x80, 0x80, 0x80, 0x80, 0x00, 0x00 },  // Frame 53
  { 0x80, 0x80, 0x80, 0x00, 0x00, 0x00 },  // Frame 54
  { 0x80, 0x80, 0x00, 0x00, 0x00, 0x00 },  // Frame 55
  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 },  // Frame 56
  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }   // Frame 57
};

/* Animation Data - HGFEDCBA Map */
const uint8_t MENU_ANIMATION[6][6] = {
  { 0x39, 0x00, 0x00, 0x00, 0x00, 0x0f },  // Frame 0
  { 0x79, 0x00, 0x00, 0x00, 0x00, 0x4f },  // Frame 1
  { 0x79, 0x40, 0x00, 0x00, 0x40, 0x4f },  // Frame 2
  { 0x39, 0x40, 0x40, 0x40, 0x40, 0x0f },  // Frame 3
  { 0x39, 0x00, 0x40, 0x40, 0x00, 0x0f },  // Frame 4
  { 0x39, 0x00, 0x00, 0x00, 0x00, 0x0f }   // Frame 5
};

uint8_t defaultMessage[6] = {0x39,0x00,0x00,0x00,0x00,0x0f};


bool autoFire = false;
int autoFireSensorReading = -1;
int autoFireInterval = AUTO_FIRE_MAX_INTERVAL;
unsigned long lastMessage = 0;
unsigned int ledBrightness = 0;

using TimerType = decltype(timer);
TimerType::Task autoFireTimerTask = 0;


void showMessage(const char* msg) {
  display.stopAnimation();
  display.showString(msg);
  lastMessage = millis();
}

void showAutoFirePercent(int percent) {
  char msg[8];
  snprintf(msg, sizeof(msg), "AF %3d", percent);
  showMessage(msg);
}

// TIMER TASK: Auto-fire action executed every autoFireInterval when enabled
bool autoFireTask(void *) {
  if (!autoFire) {
    return false;
  }

  digitalWrite(PIN_RPI_FIRE, HIGH);
  digitalWrite(PIN_BUTTON_LED_FIRE, HIGH);

  timer.in(80, [](void *) {
    digitalWrite(PIN_RPI_FIRE, LOW);
    digitalWrite(PIN_BUTTON_LED_FIRE, LOW);
    return false;
  });

  return true;
}

static void handleAfToggle() {
  Serial.print("Button Press: Auto fire (GPIO");
  Serial.print(PIN_BUTTON_TOGGLE_AF);
  Serial.println(")");

  autoFire = !autoFire;

  if (autoFire) {
    if (!autoFireTimerTask) {
      autoFireTimerTask = timer.every(autoFireInterval, autoFireTask);
    }
  } else {
    if (autoFireTimerTask) {
      timer.cancel(autoFireTimerTask);
      autoFireTimerTask = 0;
      digitalWrite(PIN_BUTTON_LED_FIRE, HIGH);
    }
  }

  display.showString(autoFire ? "AF  on" : "AF off");

  timer.in(2000, [](void *) -> bool {
    if (autoFire) {
      int percent = round(autoFireSensorReading * 100 / AUTO_FIRE_RESOLUTION);
      showAutoFirePercent(percent);
    } else {
      display.setSegments(defaultMessage, 6);
      digitalWrite(PIN_BUTTON_LED_FIRE, HIGH);
    }
    return false;
  });

}

static void handleMenuToggle() {
  Serial.print("Button Press: Menu toggle (GPIO");;
  Serial.print(PIN_BUTTON_TOGGLE_MENU);
  Serial.println(")");

  digitalWrite(PIN_RPI_TOGGLE_MENU, HIGH);
  delay(30);
  digitalWrite(PIN_RPI_TOGGLE_MENU, LOW);

  display.showAnimation(MENU_ANIMATION, FRAMES(MENU_ANIMATION), TIME_MS(75));
}

// TIMER TASK: Read the auto-fire interval from the analog sensor
bool readAutoFireInterval(void *) {
  int reading = analogRead(PIN_AUTO_FIRE_LEVEL);
  if (reading > autoFireSensorReading + 30 || reading < autoFireSensorReading - 30) {
    autoFireInterval = map(reading, 0, AUTO_FIRE_RESOLUTION, AUTO_FIRE_MIN_INTERVAL, AUTO_FIRE_MAX_INTERVAL);

    if (autoFire && autoFireTimerTask) {
      timer.cancel(autoFireTimerTask);
      autoFireTimerTask = timer.every(autoFireInterval, autoFireTask);
    }

    int percent = round(reading * 100 / AUTO_FIRE_RESOLUTION);
    if (autoFireSensorReading != -1 && autoFire) {
      showAutoFirePercent(percent);
    }
    
    autoFireSensorReading = reading;
    if (DEBUG) Serial.printf("Auto-fire @ %d \n", autoFireSensorReading);
  }

  return true;
}

// ANIMATION TASK: Rainbow cycle on the LED strip
bool rainbowAnimation(void *) {
  static long firstPixelHue = 0;
  
  if (firstPixelHue >= 5*65536) firstPixelHue = 0;

  for (int i = 0; i < led.numPixels(); i++) { 
    int pixelHue = firstPixelHue + (i * 65536L / led.numPixels());
    led.setPixelColor(i, led.gamma32(led.ColorHSV(pixelHue)));
  }

  led.show();
  if (ledBrightness < LED_MAX_BRIGHTNESS) {
    ledBrightness += 1;
    led.setBrightness(ledBrightness);
  }
  firstPixelHue += 500;

  return true;
}

bool checkAnimation(void *) {
  if (lastMessage < millis() - 5000) {
    display.showAnimation(ANIMATION, FRAMES(ANIMATION), TIME_MS(150));
  }

  return true;
}

// Start the periodic timers
void startTimers() {
  timer.every(100, readAutoFireInterval);
  timer.every(75, rainbowAnimation);
  // timer.every(1000, checkAnimation);
}

// SETUP AND LOOP
void setup() {
  pinMode(PIN_BUTTON_LED_TOGGLE_AF, OUTPUT);
  pinMode(PIN_BUTTON_LED_TOGGLE_MENU, OUTPUT);
  pinMode(PIN_BUTTON_LED_FIRE, OUTPUT);
  pinMode(PIN_RPI_TOGGLE_MENU, OUTPUT);
  pinMode(PIN_RPI_FIRE, OUTPUT);

  pinMode(PIN_BUTTON_TOGGLE_AF, INPUT_PULLDOWN);
  pinMode(PIN_BUTTON_TOGGLE_MENU, INPUT_PULLDOWN);

  if (DEBUG) Serial.begin(115200);
  if (DEBUG) Serial.println("Datort o'cade Serial ready");

  digitalWrite(PIN_BUTTON_LED_TOGGLE_AF, HIGH);
  digitalWrite(PIN_BUTTON_LED_TOGGLE_MENU, HIGH);
  digitalWrite(PIN_BUTTON_LED_FIRE, HIGH);

  led.begin();
  led.setBrightness(ledBrightness);
  led.show();

  analogReadResolution(12);

  display.begin();
  display.setScrolldelay(125);
  display.setBrightness(3);
  showMessage("Datort o'cade");
  if (DEBUG) showMessage("Debug on");
  lastMessage = 0;

  btnToggleAf.attachPress(handleAfToggle);
  btnToggleMenu.attachPress(handleMenuToggle);

  startTimers();
  display.showAnimation(ANIMATION, FRAMES(ANIMATION), TIME_MS(50));
  display.setSegments(defaultMessage, 6);
}

void loop() {
  timer.tick();
  btnToggleAf.tick();
  btnToggleMenu.tick();
}

